/*
*	replication.h (wb)
*
*	Data structure decls and prototypes relating to the replication
*	manager
*
*  Created by: Win Bausch and Bettina Kemme
*
*/


#ifndef RMGR_H
#define RMGR_H


//#define RMGR_DEBUG
//#define RMGR_STAT

#include <unistd.h>

#include "c.h"
#include "libpq/bufferedSock.h"
#include "replication/writeset.h"


//#define ROUTING_HDR_SIZE 4
#define RMGR_HDR_SIZE 8
#define RMGR_ID_SIZES 8   /* 2* sizeof(uint32) */
#define BKND_ID_SIZE 8
#define GROUPCOMM_HDR_SIZE RMGR_HDR_SIZE + BKND_ID_SIZE

#define RMGR_PORTNUM 5430

/*
*	supported msg types
*/
typedef enum rc_msg_t{
	MSG_OPENING,
	MSG_CLOSING,
	MSG_NEW_TXN,
	MSG_WRITESET,
	MSG_WS_RECEIVED,
	MSG_LOCKED,
	MSG_COMMIT,
	MSG_ABORT,
	MSG_ACK,
	MSG_READY,
	MSG_PROTO_ERROR,
	MSG_INVAL,
} rc_msg_t;

/*
* valid connection states (local and remote txn)
*/
typedef enum rc_state_t{
	CLOSED,
	READ_PHASE,
	QUERY_PHASE,
	SEND_PHASE,
	LOCK_PHASE,
	WRITE_PHASE,
/*	COMMITTABLE,*/    /* now in result */
	READY_TO_CLOSE,   /* bk for remote backends */
	CLOSE_AND_DESTROY,  /* bk: for local backends */
	ABORTED,
	NO_BKND,
	BKND_PRESENT,
} rc_state_t;


/* bk
*  local process connects once (when first write set is sent)
*  and then reuses connection */
typedef enum rc_connection_t {
  NOT_YET_CONNECTED,
  CONNECTED
} rc_connection_t;


/* bk
*  add result, determined by the the msg_commit/msg_abort
*/
typedef enum rc_result_t{
  UNKNOWN,
  COMMITTABLE,
  TO_ABORT
} rc_result_t;


/*
*	message receiving states
*/
typedef enum recv_state_t{
	RECV_IDLE,
	RECV_HDR,
	RECV_DATA,
} recv_state_t;

/*
*	routing targets (src and dest)
*
*	ATTN: order is important, as these values 
*		are used as indexes into an array!!!!
*/
typedef enum route_addr_t{
	LOCAL,		
	REMOTE,
	GROUPCOMM,
	INVAL,
}route_addr_t;

/*
*	struct used in message receiving process
*	bsock - buffered socket
*	recv_state - contains state info about which part of
*			a message we are receiving (header or body)
*	datalen - length of the data chunk we wait for
*
*	see RmgrReceive in replicaManager.c for an example of its use	
*/
typedef struct DataConn{
	bufsockptr		bsock;
	recv_state_t 	recv_state;
	int 			datalen;
} DataConn;

/*
*	Rmgr uses this struct to manage a backend
*	that connected to send its write set
*
*	dc - message interface to the backend
*	state - the state of the connection
*	hostid - the hostid associated to the txn
*	xid - the xid associated to the txn
*	numDelivered - number of messages that have
*				  been delivered, determines if
*				  any msgs in transit 
*	
*	NOTE: hostid and xid are used as key to 
*		identify the backends belonging to 
*		a replicated txn. In other words, a 
*		local transaction and all its remote
*		txns carry the same key: the xid of 
*		the local txn plus the host id of 
*		the machine where the local txn runs
*/
typedef struct Rconn{
	DataConn		dc;	
	rc_state_t		state;
	rc_result_t		result;
	int				numDelivered;
	int				procid;
	uint32			hostid;
	uint32			xid;
#ifdef RMGR_STAT
	clock_t			lastStop;
	int				statSlot_ind;
#endif
} Rconn;

/*
*	pg_transis.c (main routine)
*/
/* now new two functions */
extern int GroupCommMain_basic(int rmgrsock, int rmgrsock_basic);
extern int GroupCommMain_total(int rmgrsock, int rmgrsock_total);
/*
*	replicaManager.c 
*/
extern int RmgrMain(int gcsock_total_send, int gcsock_basic_send, int gcsock_total, int gcsock_basic);
extern void ReplicaMgrCleanUp(int dummy);
extern void HandleProtoError(Rconn *conn);

/*
*	rmgrLib.c
*/
extern void RmgrInitConn(bufsockptr *rmgrSock);
extern void RmgrDestroyConn(int dummy, bufsockptr rmgrSock);
extern int RmgrConnect(bufsockptr rmgrSock, int ReplicaMgrPort, int procid, int hostid, int xid);
extern int RmgrDisconnect(bufsockptr rmgrSock);
extern int RmgrSendWriteSet(WriteSetPtr ws, bufsockptr rmgrSock);
extern int RmgrSendTxnId(bufsockptr rmgrSock, int procid, int xid);
extern int RmgrReceiveWriteSet(WriteSetPtr ws, bufsockptr rmgrSock);
extern int RmgrSendMsg(rc_msg_t msg, bufsockptr rmgrSock, bool waitForAck);
extern int RmgrReceiveMsg(rc_msg_t *msg, bufsockptr rmgrSock, bool blockingRead);

/*
*	perfmeasure.c
*/
extern int LogFile_init(void);
extern int LogFile_record(bool isLocal, uint32 xid, const char *recordTag);
extern int LogFile_close(void);

#endif
