/*
 * tcpwrap.c
 * Copyright (C) 2000  Shugo Maeda <shugo@ruby-lang.org>
 *
 * This file is part of ruby-tcpwrap.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA
 */

#include "ruby.h"
#include "rubyio.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <tcpd.h>
#ifdef HAVE_IDENT_H
#include <ident.h>
#endif

#define DEFAULT_RFC1413_QUERY_TIMEOUT 30

int allow_severity = 0;
int deny_severity = 0;

static VALUE cTCPServer;
static VALUE eSocket;
static VALUE cTCPWrapper;

typedef struct tcp_wrapper {
    VALUE daemon;
    VALUE server;
    int ident_lookup;
    int ident_timeout;
} tcp_wrapper_t;

static void mark_tcpd(tcp_wrapper_t *tcpd)
{
    rb_gc_mark(tcpd->daemon);
    rb_gc_mark(tcpd->server);
}

static void free_tcpd(tcp_wrapper_t *tcpd)
{
    free(tcpd);
}

static VALUE tcpd_s_new(int argc, VALUE *argv, VALUE self)
{
    VALUE obj, daemon, server, ident_lookup, ident_timeout;
    int to;
    tcp_wrapper_t *tcpd;

    rb_scan_args(argc, argv, "22", &daemon, &server, &ident_lookup, &ident_timeout);
    if (argc == 4) {
	to = NUM2INT(ident_timeout);
    }
    else {
	to = DEFAULT_RFC1413_QUERY_TIMEOUT;
    }
    Check_Type(daemon, T_STRING);
    if (!rb_obj_is_kind_of(server, cTCPServer))
	rb_raise(rb_eTypeError, "expecting TCPServer");
    obj = Data_Make_Struct(cTCPWrapper, tcp_wrapper_t, mark_tcpd, free_tcpd, tcpd);
    tcpd->daemon = daemon;
    tcpd->server = server;
    tcpd->ident_lookup = RTEST(ident_lookup);
    tcpd->ident_timeout = to;
    return obj;
}

static VALUE tcpd_accept(VALUE self)
{
    int access = 0;
    tcp_wrapper_t *tcpd;
    VALUE sock;
    int sockfd;
    OpenFile *fptr;
    struct sockaddr_in addr;
    struct hostent *h;
    char *client_name = NULL;
    char *client_addr = NULL;
    char *client_user = NULL;
    int len = sizeof(addr);

    Data_Get_Struct(self, tcp_wrapper_t, tcpd);
 again:
    sock = rb_funcall(tcpd->server, rb_intern("accept"), 0);
    GetOpenFile(sock, fptr);
    sockfd = fileno(fptr->f);
    if (getpeername(sockfd, (struct sockaddr*) &addr, &len) < 0)
	rb_sys_fail("getpeername(2)");
    client_addr = inet_ntoa(addr.sin_addr);
    h = gethostbyaddr((char *)&addr.sin_addr,
		      sizeof(addr.sin_addr),
		      addr.sin_family);
    if (h == NULL) {
#ifdef HAVE_HSTERROR
	extern int h_errno;
	rb_raise(eSocket, "%s", (char *) hsterror(h_errno));
#else
	rb_raise(eSocket, "host not found");
#endif
    }
    client_name = h->h_name;
#ifdef HAVE_IDENT_ID
    if (tcpd->ident_lookup)
	client_user = ident_id(sockfd, tcpd->ident_timeout);
#endif
    if (!hosts_ctl(RSTRING(tcpd->daemon)->ptr,
		   (client_name == NULL) ? STRING_UNKNOWN : client_name,
		   (client_addr == NULL) ? STRING_UNKNOWN : client_addr,
		   (client_user == NULL) ? STRING_UNKNOWN : client_user)) {
	rb_funcall(sock, rb_intern("shutdown"), 0);
	rb_funcall(sock, rb_intern("close"), 0);
#ifdef HAVE_IDENT_ID
	if (client_user)
	    free(client_user);
#endif
	goto again;
    }
#ifdef HAVE_IDENT_ID
	if (client_user)
	    free(client_user);
#endif
    return sock;
}

void Init_tcpwrap()
{
    rb_require("socket");
    cTCPServer = rb_const_get(rb_cObject, rb_intern("TCPServer"));
    eSocket = rb_const_get(rb_cObject, rb_intern("SocketError"));
    cTCPWrapper = rb_define_class("TCPWrapper", rb_cObject);
    rb_define_singleton_method(cTCPWrapper, "new", tcpd_s_new, -1);
    rb_define_method(cTCPWrapper, "accept", tcpd_accept, 0);
}
