/* ActionRepeater.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.AbstractButton;
import javax.swing.ButtonModel;
import javax.swing.Timer;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/**
 * Provides a means of repeating an action as long as a certain button is pressed.
 * This class must be extended to provide an implementation of {@link #doAction}
 * and possibly of {@link #buttonFirstPressed} and {@link #buttonPressCanceled}.
 * An object of that type can then be registered with a button
 * using the {@link #registerWith} method.<p>
 * When the button is first pressed, the method {@code buttonFirstPressed} is called, followed
 * by {@code doAction}.
 * Then, while the button remains armed, {@code doAction} is called again and again.
 * When the button press is canceled (i.e., when the mouse button is released while the button is
 * not armed) a final call to {@code buttonPressCanceled} is invoked. A 'normal' button press is
 * handled by the action listeners of the button, and not by this class.<p>
 * In the following example an extension of this class is used to implement a
 * 'zoom' button which repeatedly zooms in while the button is pressed. Releasing
 * the button while not armed, resets the zoom factor to 1.
 * <pre>
 *    class ZoomRepeater extends ActionRepeater {
 *          private double zoomFactor;
 *
 *          public void buttonFirstPressed () {
 *              zoomFactor = 1.0;
 *          }
 *          
 *          public void doAction () {
 *              zoomFactor *= 1.2;
 *              setZoom (zoomFactor);
 *          }
 *            
 *          public void buttonPressCanceled () {
 *              setZoom (1.0);
 *          }
 *          ...
 *    }
 * </pre>
 * An object of this class can be attached to the corresponding
 * button as follows:
 * <pre>
 *    new ZoomRepeater().registerWith (zoomButton);
 * </pre>
 * This makes <code>zoomButton</code> behave as requested.
 */
public abstract class ActionRepeater implements ChangeListener, ActionListener {
    
    /**
     * Set the interval between succesive calls to {@link #doAction}.
     * @param interval new interval in milliseconds
     */
    public void setInterval(int interval) {
        timer.setDelay(interval);
    }
    
    //
    private final Timer timer;
    
    /**
     * Create a new object of this type. Will repeatedly 'fire' every 150 ms.
     */
    public ActionRepeater() {
        this.timer = new Timer(150, this);
    }
    
    //
    private AbstractButton button;
    
    /**
     * Button to which this repeater is currently registered, or {@code null}
     * if none.
     */
    public AbstractButton getButton() {
        return this.button;
    }
    
    /**
     * Register this repeater with a button. A single repeater can be registered
     * with at most a single button at the same time. Registering with a new button
     * will automatically undo an earlier registration with another button.
     * @param button Button to which this repeater should be registered, or {@code null}
     * to unregister.
     */
    public void registerWith(AbstractButton button) {
        if (this.button != null)
            button.removeChangeListener(this);
        this.button = button;
        if (this.button != null)
            button.addChangeListener(this);
    }
    
    /**
     * Action which is performed repeatedly while the button
     * is pressed.
     */
    public abstract void doAction();
    
    /**
     * Method which is called when the button is first pressed.<p>
     * This implementation is empty.
     */
    public void buttonFirstPressed() {
        // empty
    }
    
    /**
     * Method which is called when the button press is canceled: i.e.,
     * when the mouse button is released when the button is not armed.<p>
     * This implementation is empty.
     */
    public void buttonPressCanceled() {
        // empty
    }
    
    /**
     * Listens to changes in the state of the button to which this object is 
     * registered. Subclasses should override {@link #doAction}, 
     * {@link #buttonFirstPressed} and {@link #buttonPressCanceled}
     * instead of this method.
     */
    public void stateChanged(ChangeEvent e) {
        assert e.getSource() == button;
        ButtonModel model = button.getModel();
        if (model != null) {
            if (model.isPressed()) {
                if (!timer.isRunning()) {
                    buttonFirstPressed();
                    doAction();
                    timer.start();
                }
            } else if (timer.isRunning()) {
                timer.stop();
                if (!model.isArmed())
                    buttonPressCanceled();
            }
        }
    }
    
    /**
     * Called whenever the timer fires. Subclasses should override {@link #doAction} 
     * instead of this method.
     */
    public void actionPerformed(ActionEvent e) {
        doAction();
    }
    
}
