#! /bin/bash

# debi:  Install current version of deb package
# debit: Install and test current version of deb package
# debc:  List contents of current version of deb package
#
# debi and debc originally by Christoph Lameter <clameter@debian.org>
# Copyright Christoph Lameter <clameter@debian.org>
# debit originally by Jim Van Zandt <jrv@vanzandt.mv.com>
# Copyright 1999 Jim Van Zandt <jrv@vanzandt.mv.com>
# Modifications by Julian Gilbey <jdg@debian.org>, 1999
# Copyright 1999 by Julian Gilbey
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# Non-help options:
#  -a<arch>    Debian architecture
#  -t<type>    GNU machine type
# These are used to determine the name of the .changes file.

set -e

PROGNAME=`basename $0`

usage () {
    if   [ "$PROGNAME" = debi ];  then usage_i
    elif [ "$PROGNAME" = debit ]; then usage_it
    elif [ "$PROGNAME" = debc ];  then usage_c
    else echo "Unrecognised invocation name: $PROGNAME" >&2; exit 1
    fi;
}

usage_i () {
    echo \
"Usage: $PROGNAME [options] [.changes file]
  Install the .deb file(s) just created, as listed in the generated
  .changes file or the .changes file specified.
  Options:
    -a<arch>        Search for .changes file made for Debian build <arch>
    -t<target>      Search for .changes file made for GNU <target> arch
    --help          Show this message
    --version       Show version and copyright information"
}

usage_it () {
    echo \
"Usage: $PROGNAME [options] [.changes file]
  Install and test current version of deb package
  Options:
    -a<arch>        Search for changes file made for Debian build <arch>
    -t<target>      Search for changes file made for GNU <target> arch
    --help          Show this message
    --version       Show version and copyright information"
}

usage_c () {
    echo \
"Usage: $PROGNAME [options] [.changes file]
  Display the contents of the .deb file(s) just created, as listed
  in the generated .changes file or the .changes file specified.
  Options:
    -a<arch>        Search for changes file made for Debian build <arch>
    -t<target>      Search for changes file made for GNU <target> arch
    --help          Show this message
    --version       Show version and copyright information"
}

version () { echo \
"This is $PROGNAME, from the Debian devscripts package, version ###VERSION###
This code is copyright 1999 by Julian Gilbey, all rights reserved.
Based on original code by Christoph Lameter and James R. Van Zandt.
This program comes with ABSOLUTELY NO WARRANTY.
You are free to redistribute this code under the terms of
the GNU General Public License, version 2 or later."
}

mustsetvar () {
    if [ "x$2" = x ]
    then
	echo >&2 "$0: unable to determine $3"
	exit 1
    else
	# echo "$0: $3 is $2"
	eval "$1=\"\$2\""
    fi
}

while [ $# != 0 ]
do
    value="`echo x\"$1\" | sed -e 's/^x-.//'`"
    case "$1" in
	-a*)	targetarch="$value" ;;
	-t*)	targetgnusystem="$value" ;;
	--help)	usage; exit 0 ;;
	--version)
		version; exit 0 ;;
	-*)	echo "Unrecognised option: $1" >&2
		usage >&2; exit 1 ;;
	*)	break ;;
    esac
    shift
done

case $# in
    0)	# Look for .changes file via debian/changelog
	until [ -f debian/changelog ]; do
	    cd ..
	    if [ `pwd` = "/" ]; then
		echo "Cannot find debian/changelog anywhere." >&2
		exit 1
	    fi
	done

	mustsetvar package "`dpkg-parsechangelog | sed -n 's/^Source: //p'`" \
	    "source package"
	mustsetvar version "`dpkg-parsechangelog | sed -n 's/^Version: //p'`" \
	    "source version"

	if ! command -v dpkg-architecture > /dev/null 2>&1; then
	    echo "This program depends on dpkg-architecture; your dpkg is far too old" >& 1
	    exit 1
	fi

	mustsetvar arch "`dpkg-architecture -a${targetarch} -t${targetgnusystem} -qDEB_HOST_ARCH`" "build architecture"

	sversion=`echo "$version" | perl -pe 's/^\d+://'`
	pva="${package}_${sversion}_${arch}"
	changes="../$pva.changes"
	;;

    1)	changes=$1 ;;
    *)	echo "Only one .changes file may be specified!  Try $PROGNAME --help for more help" >&2
	exit 1 ;;
esac

cd `dirname $changes`
changes=`basename $changes`

if [ ! -r "$changes" ]
then echo "Can't read $changes!" >&2; exit 1; fi

debs="`perl -e '
@debs = ();
while (<>) {
    last if $infiles and /^[^ ]/;
    /^Files:/ and $infiles=1, next;
    next unless $infiles;
    / (\S*.deb)$/ and push @debs, $1;
}
print join(" ",@debs)' $changes`"

if [ "$PROGNAME" = debi -o "$PROGNAME" = debit ]
then
    debpkg -i $debs
else
    # debc
    for deb in $debs
    do
	echo $deb
	echo $deb | tr -c -- '-\012' -
	dpkg-deb -I $deb
	dpkg-deb -c $deb
    done
fi

# Extra bits if we are running debit
if [ "$PROGNAME" = debit ]
then
    for deb in $debs; do
	PACKAGE=`echo "$deb" | sed -e 's/_.*//'`
	if [ -e /usr/lib/debian-test/tests/$PACKAGE -o \
		-e /usr/lib/debian-test/default-tests/$PACKAGE ]; then
	    echo "testing $PACKAGE..."
	    debian-test -v $PACKAGE
	else
	    echo "testing $PACKAGE... (no test available)"
	fi
    done
fi
