package com.ibm.bsf.engines.jpython;

import java.util.Vector;

import org.python.util.*;
import org.python.core.*;

import com.ibm.bsf.*;
import com.ibm.bsf.util.BSFEngineImpl;
import com.ibm.bsf.util.BSFFunctions;

/**
 * This is the interface to JPython (http://www.jpython.org/) from BSF.
 *
 * @author   Sanjiva Weerawarana
 * @author   Finn Bock <bckfnn@worldonline.dk>
 */

public class JPythonEngine extends BSFEngineImpl {
  PythonInterpreter interp;

  /**
   * Initialize the engine.
   */
  public void initialize (BSFManager mgr, String lang,
                          Vector declaredBeans) throws BSFException {
    super.initialize (mgr, lang, declaredBeans);

    // create an interpreter
    interp = new PythonInterpreter ();

    // register the mgr with object name "bsf"
    interp.set ("bsf", new BSFFunctions (mgr, this));

    int size = declaredBeans.size ();
    for (int i = 0; i < size; i++) {
      declareBean ((BSFDeclaredBean) declaredBeans.elementAt (i));
    }
  }

  /**
   * Evaluate an expression.
   */
  public Object eval (String source, int lineNo, int columnNo, 
		      Object script) throws BSFException {
    try {
      Object result = interp.eval (script.toString ());
      if (result != null && result instanceof PyJavaInstance)
        result = ((PyJavaInstance)result).__tojava__(Object.class);
      return result;
    } catch (PyException e) {
      e.printStackTrace ();
      throw new BSFException (BSFException.REASON_EXECUTION_ERROR,
			      "exception from JPython: " + e, e);
    }
  }

  /**
   * Execute a script. 
   */
  public void exec (String source, int lineNo, int columnNo,
		    Object script) throws BSFException {
    try {
      interp.exec (script.toString ());
    } catch (PyException e) {
      e.printStackTrace ();
      throw new BSFException (BSFException.REASON_EXECUTION_ERROR,
			      "exception from JPython: " + e, e);
    }
  }

  /**
   * call the named method of the given object.
   */
  public Object call (Object object, String method, Object[] args) 
                                                        throws BSFException {
    
    PyObject[] pyargs = Py.EmptyObjects;
    if (args != null) {
      pyargs = new PyObject[args.length];
      for (int i = 0; i < pyargs.length; i++)
        pyargs[i] = Py.java2py(args[i]);
    }

    if (object != null) {
      PyObject o = Py.java2py(object);
      return unwrap(o.invoke(method, pyargs));
    }
    PyObject m = interp.get(method);
    if (m == null)
        m = interp.eval(method);
    if (m != null) {
	return unwrap(m.__call__(pyargs));
    }
    return null;
  }

  public Object unwrap(PyObject result) {
    if (result != null) {
       Object ret = result.__tojava__(Object.class);
       if (ret != Py.NoConversion)
          return ret;
    }
    return result;
  }


  /**
   * Declare a bean
   */
  public void declareBean (BSFDeclaredBean bean) throws BSFException {
    interp.set (bean.name, bean.bean);
  }

  /**
   * Undeclare a previously declared bean.
   */
  public void undeclareBean (BSFDeclaredBean bean) throws BSFException {
    interp.set (bean.name, null);
  }
}
