/**
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package javax.management.openmbean;

import java.io.Serializable;

import javax.management.MBeanOperationInfo;
import javax.management.MBeanParameterInfo;

import java.util.Arrays;



/**
 * Describes an operation of an <code>OpenMBean</code>
 *
 * @author <a href="mailto:shadow12@users.sourceforge.net">Bronwen Cassidy</a>
 * @author <a href="mailto:shadow12@users.sourceforge.net">Warren Mira</a>
 * @version $Revision: 1.8 $
 */
public class OpenMBeanOperationInfoSupport extends MBeanOperationInfo implements OpenMBeanOperationInfo, Serializable
{
	private static final long serialVersionUID = 4996859732565369366L;

	private OpenType returnOpenType;

    private transient int hashCode = 0;
    private transient String toStringName = null; 

    /**
     * Creates an instance of <code>OpenMBeanOperationInfoSupport</code>
     * 
     * @param name The name of this operation
     * @param description The description
     * @param signature The parameter[s] of this operation
     * @param returnOpenType The return type
     * @param impact The impact of the operation
     * @see OpenMBeanOperationInfo#getImpact
     *
     * @throws ArrayStoreException If signature is not of type {@link MBeanParameterInfo }
     *
     */
	public OpenMBeanOperationInfoSupport(
		String name,
		String description,
		OpenMBeanParameterInfo[] signature,
		OpenType returntype,
		int impact)
	{
		super(
			name,
			description,
			(signature == null)
				? (MBeanParameterInfo[]) Arrays.asList(
					new OpenMBeanParameterInfo[0]).toArray(
					new MBeanParameterInfo[0])
				: (MBeanParameterInfo[]) Arrays.asList(signature).toArray(
					new MBeanParameterInfo[0]),
			returntype == null ? "" : returntype.getClassName(),
			impact);

		// super class constructors don't do the necessary validation
		if (name == null || name.length() == 0)
		{
			throw new IllegalArgumentException("name cannot be null or empty");
		}

		if (description == null || description.length() == 0)
		{
			throw new IllegalArgumentException("descripiton cannot be null or empty");
		}

		if (returntype == null)
		{
			throw new IllegalArgumentException("return open type cannont be null");
		}

		if (impact != MBeanOperationInfo.ACTION
			&& impact != MBeanOperationInfo.ACTION_INFO
			&& impact != MBeanOperationInfo.INFO
			&& impact != MBeanOperationInfo.UNKNOWN)
		{
			throw new IllegalArgumentException("invalid impact");
		}

		if (signature != null
			&& signature.getClass().isInstance(MBeanParameterInfo[].class))
		{
			throw new ArrayStoreException("signature elements can't be assigned to MBeanParameterInfo");
		}

		this.returnOpenType = returntype;
	}


    /**
     * Returns the returntype as an <code>OpenType</code>
     *
     * @return OpenType The OpenType instance 
     *
     */
	public OpenType getReturnOpenType()
	{
		return returnOpenType;
	}

    /**
     * Test the specified object for equality.
     *
     * <p> 
     * This method will return  true if and only if the following 
     * conditions are true:
     * </p>
     * <p>
     * <ul>
     *      <li>obj is not null</li>
     *      <li>obj also implements OpenMBeanOperationInfo</li>
     *      <li>their names are equal</li>
     *      <li>their signatures are equal</li>
     *      <li>their return opentypes are equal</li>
     *      <li>their impacts are equal</li>
     * </ul>
     *
     * </p>
     *
     * @param obj The object being compared to
     * @return boolean 
     *
     */
	public boolean equals(Object obj)
	{
        if ( obj == null ) return false;
        if ( obj == this ) return true;

        //better if we wont allow throwing exception
        if ( obj instanceof OpenMBeanOperationInfo  )
        {
            OpenMBeanOperationInfo other = (OpenMBeanOperationInfo)obj;
            if  ((getName() == null && other.getName() == null) ||
                (getName() != null && getName().equals(other.getName())))
            {
                if ( other.getImpact() !=  getImpact() ) return false;
                if ( getReturnOpenType() == null ) 
                {
                    if ( other.getReturnOpenType() != null ) return false;
                } 
                else 
                {
                    if ( !getReturnOpenType().equals(other.getReturnOpenType()) ) return false;
                }


                if ( !Arrays.equals(getSignature(), other.getSignature())) return false;
                return true;

            }

        }
        return false;
 
	}

    /**
     * Returns the hashcode of this <code>OpenMBeanOperationInfo</code>
     *  
     * @return int The hashcode
     *
     */
	public int hashCode()
	{

    
        if ( hashCode == 0 ) 
        {
			int result = getName().hashCode();
			result += returnOpenType.hashCode();
			result += getImpact();
			result += java.util.Arrays.asList(getSignature()).hashCode();
			hashCode = result;
        }
        return hashCode; 
	}


    /**
     * Returns a String representation of this OpenMBeanOperationInfoSupport
     *
     * @return String The string representation
     *
     */
	public String toString()
	{
        if ( toStringName == null ) 
        {
            StringBuffer sb = new StringBuffer();
            sb.append(getClass().getName());
            sb.append("(name="); 
            sb.append(getName()); 
            sb.append(",signature="); 
            sb.append(java.util.Arrays.asList(getSignature()).toString()); 
            sb.append(",returnOpenType="); 
            sb.append(returnOpenType.toString()); 
            sb.append(",impact="); 
            sb.append(getImpact()); 
            sb.append(")"); 
   
            toStringName = sb.toString();
        }     
        
        return toStringName;
    
	}
    //private

    private static MBeanParameterInfo[] createParameterInfo(OpenMBeanParameterInfo[] signature) 
    {

        if ( signature == null ) return null;
        
        MBeanParameterInfo[] paramInfo =
            new MBeanParameterInfo[signature.length];

        System.arraycopy(signature,0,paramInfo,0,paramInfo.length);

        return paramInfo; 


    } //end
}
