/**
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package javax.management.openmbean;

import java.io.Serializable;
import java.util.Arrays;

import javax.management.MBeanConstructorInfo;
import javax.management.MBeanParameterInfo;

/**
 *
 * @author <a href="mailto:shadow12@users.sourceforge.net">Bronwen Cassidy</a>
 * @version $Revision: 1.3 $
 */

/**
 * Describes a constructor of an openMBean
 */
public class OpenMBeanConstructorInfoSupport extends MBeanConstructorInfo implements OpenMBeanConstructorInfo, Serializable
{
	private static final long serialVersionUID = -4400441579007477003L;

	// No non-transient fields allowed
	private transient int m_hashcode = 0;

	/**
	 * <p>Constructs an OpenMBeanConstructorInfoSupport instance, which describes the constructor of a class of open MBeans with the specified name, description and signature</p>
	 * <p>The signature array parameter is internally copied, so that subsequent changes to the array referenced by signature have no effect on this instance</p>
	 *
	 * @param name - cannot be a null or empty string
	 * @param description - cannot be a null or empty string
	 * @param signature - can be null or empty if there are no parameters to describe
	 *
	 * @throws IllegalArgumentException - if name or description are null or empty string
	 * @throws ArrayStoreException - If signature is not an array of instances of a subclass of MBeanParameterInfo
	 */
	public OpenMBeanConstructorInfoSupport(String name, String description, OpenMBeanParameterInfo[] signature)
	{
		super(
			name,
			description,
			signature == null
				? null
				: (MBeanParameterInfo[]) Arrays.asList(signature).toArray(new MBeanParameterInfo[0]));
		if(name == null || name.trim().equals("")) throw new IllegalArgumentException("name parameter cannot be null or an empty string");
		if(description == null || description.trim().equals("")) throw new IllegalArgumentException("description parameter cannot be null or an empty string");
	}

	/**
	 * Tests if the object given in the parameter is equal to this instance
	 *
	 * @param obj the object to test for equality
	 * @return true if equals false otherwise
	 */
	public boolean equals(Object obj)
	{
		if(!(obj instanceof OpenMBeanConstructorInfo)) return false;
		OpenMBeanConstructorInfo toCompare = (OpenMBeanConstructorInfo)obj;
        return (getName().equals(toCompare.getName()) && Arrays.equals(getSignature(), toCompare.getSignature()));
	}

	/**
	 * @return the calculated hashCode for this instance. The same values tested for in equals are used to calculate the hashCode
	 */
	public int hashCode()
	{
		if(m_hashcode == 0)
		{
			int result = getName().hashCode();
			result += Arrays.asList(getSignature()).hashCode();
			m_hashcode = result;
		}
		return m_hashcode;
	}

	/**
	 * @return the human readable representation of this instance
	 */
	public String toString()
	{
		return (getClass().getName() + " ( name = " + getName() + " signature = " + Arrays.asList(getSignature()).toString() + " )");
	}


	/**
	 * fullfills the constructor contract that the signature array parameter is internally copied, so that subsequent changes to the array referenced by signature have no effect on this instance
	 * the ArrayStoreException is thrown by the System.arraycopy method if an element in the <code>src</code>
     * array could not be stored into the <code>dest</code> array because of a type mismatch.
	 */
    private static MBeanParameterInfo[] copyParameterInfo(OpenMBeanParameterInfo[] signature) throws ArrayStoreException
	{
	    MBeanParameterInfo[] parameterInfo = new MBeanParameterInfo[signature.length];
		System.arraycopy(signature, 0, parameterInfo, 0, signature.length);
		return parameterInfo;
	}
}
